//
//  GeometryGamesRenderer.swift
//
//  Created by Jeff on 2/10/20.
//  Copyright © 2020 Jeff Weeks. All rights reserved.
//

import MetalKit


class GeometryGamesRenderer {

	let itsModelData: GeometryGamesModel
	let itsDevice: MTLDevice
	let itsCommandQueue: MTLCommandQueue
	let itsColorPixelFormat: MTLPixelFormat
	let itsDepthPixelFormat: MTLPixelFormat	//	= MTLPixelFormat.invalid if no depth buffer is needed
	let itsSampleCount: UInt				//	typically 4 (for multisampling) or 1 (for single-sampling)
	let itsOpaqueClearColor: MTLClearColor
	let itsTransparentClearColor: MTLClearColor = MTLClearColor(red: 0.0, green: 0.0, blue: 0.0, alpha: 0.0)
	
	
	required init?(
		modelData: GeometryGamesModel,
		device: MTLDevice,
		colorPixelFormat: MTLPixelFormat,
		depthPixelFormat: MTLPixelFormat,	//	= MTLPixelFormat.invalid if no depth buffer is needed
		sampleCount: UInt,					//	typically 4 (for multisampling) or 1 (for single-sampling)
		clearColor: MTLClearColor
	) {
		itsModelData = modelData

		itsDevice = device
		itsColorPixelFormat = colorPixelFormat
		itsDepthPixelFormat = depthPixelFormat
		itsSampleCount = sampleCount
		itsOpaqueClearColor = clearColor

		guard let theCommandQueue = device.makeCommandQueue() else {
			return nil
		}
		itsCommandQueue = theCommandQueue
	}


// MARK: -
// MARK: Draw onscreen

	func render(drawable: CAMetalDrawable) {
			
		//	See
		//
		//		https://developer.apple.com/documentation/metal/presentation_objects/obtaining_presenting_and_releasing_drawables
		//
		//	for a detailed explanation of how best to organize this rendering code.
		
		//	If we need to do any compute or offscreen work on the GPU,
		//	encode and submit it here.
		
		//	Locate the drawable's MTLTexture.
		let theColorBuffer = drawable.texture

		//	Create a command buffer for the onscreen render pass.
		//	Be sure to do this before waiting on the semaphore,
		//	because if we return prematurely the semaphore won't get signaled.
		guard let theCommandBuffer = itsCommandQueue.makeCommandBuffer() else {
			print("Failed to make theCommandBuffer")
			return
		}

		//	Create a render target using theColorBuffer.
		guard let theRenderPassDescriptor = CreateRenderTarget(
			device: itsDevice,
			colorBuffer: theColorBuffer,
			sampleCount: itsSampleCount,
			clearColor: itsOpaqueClearColor,
			depthPixelFormat: itsDepthPixelFormat
		) else {
			print("Failed to make theRenderPassDescriptor")
			return
		}

		//	Let the subclass encode an onscreen render pass.
		encodeCommands(
			commandBuffer: theCommandBuffer,
			renderPassDescriptor: theRenderPassDescriptor,
			frameWidth:  UInt(theColorBuffer.width ),
			frameHeight: UInt(theColorBuffer.height),
			quality: .animation)

		//	Ask Metal to present our results for display.
		theCommandBuffer.present(drawable)

		//	Commit theCommandBuffer to the GPU.
		theCommandBuffer.commit()
	}


// MARK: -
// MARK: Draw offscreen

	func createOffscreenImage(
		widthPx: UInt,
		heightPx: UInt,
		withTransparentBackground: Bool
	) -> CGImage? {
	
		//	Create the command buffer.
		guard let theCommandBuffer = itsCommandQueue.makeCommandBuffer() else {
			return nil
		}

		//	Create an offscreen color buffer.
		guard let theColorBuffer = CreateOffscreenColorBuffer(
			device: itsDevice,
			colorPixelFormat: itsColorPixelFormat,
			widthPx: widthPx,
			heightPx: heightPx
		) else {
			return nil
		}

		//	Create a render target using theColorBuffer.
		guard let theRenderPassDescriptor = CreateRenderTarget(
			device: itsDevice,
			colorBuffer: theColorBuffer,
			sampleCount: itsSampleCount,
			clearColor: (withTransparentBackground ? itsTransparentClearColor : itsOpaqueClearColor),
			depthPixelFormat: itsDepthPixelFormat
		) else {
			return nil
		}
		
		//	Encode the offscreen render pass.
		encodeCommands(
			commandBuffer: theCommandBuffer,
			renderPassDescriptor: theRenderPassDescriptor,
			frameWidth:  UInt(theColorBuffer.width ),
			frameHeight: UInt(theColorBuffer.height),
			quality: .export)

		//	Draw the scene.
		theCommandBuffer.commit()
		theCommandBuffer.waitUntilCompleted()	//	This blocks!

		//	theColorBuffer is a MTLTexture created with a pixel format
		//
		//		MTLPixelFormat.bgr10_xr_srgb	(iDevice, opaque background)
		//		MTLPixelFormat.bgra10_xr_srgb	(iDevice, transparent background)
		//		MTLPixelFormat.rgba16Float		(simulator or catalyst)
		//
		//	Even though a color buffer of format bgr10_xr_srgb or bgra10_xr_srgb
		//	stores gamma-encoded color components, the _srgb suffix
		//	asks that the values be automatically gamma-decoded when read.
		//	So when we call
		//
		//		CIImage(mtlTexture: theColorBuffer, options: nil)
		//
		//	theColorBuffer automatically decodes its gamma-encoded values
		//	and reports linearized values to the CIImage.
		//	For consistency, we must mark theCIContext's workingColorSpace
		//	as extendedLinearSRGB, because linearized values are what it will contain.
		//	(Caution:  I have no proof that that's how Core Image works,
		//	but it seems plausible.)
		//
		//	Even when using rgba16Float in Catalyst, the exported colors
		//	come out correct -- in the Display P3 color space
		//	with no gamma-encoding problems -- for reasons I don't fully understand.
		//	In any case, I don't plan to release SwiftUI versions
		//	of the Geometry Games apps for legacy Macs, so
		//	the publicly released version will never use rgba16Float.
		//
		guard let theWorkingColorSpace = CGColorSpace(name: CGColorSpace.extendedLinearSRGB) else {
			return nil
		}
		guard let theOutputColorSpace = CGColorSpace(name: CGColorSpace.displayP3) else {
			return nil
		}

		//	Use a half-precision (16-bit) floating-point pixel format,
		//	to accommodate values below 0.0 and above 1.0,
		//	as extended sRGB coordinates require.
		//
		let thePixelFormat = CIFormat.RGBAh
		let theOptions: [CIContextOption : Any] = [
						CIContextOption.workingColorSpace: theWorkingColorSpace,
						CIContextOption.workingFormat:     thePixelFormat
		]
		let theCIContext = CIContext(options: theOptions)
		guard let theCIImage = CIImage(mtlTexture: theColorBuffer, options: nil) else {
			return nil
		}
		let theFlippedCIImage = theCIImage.oriented(CGImagePropertyOrientation.downMirrored)
		let theRect = CGRect(x: 0, y: 0, width: Int(widthPx), height: Int(heightPx))
		guard let theCGImage = theCIContext.createCGImage(
										theFlippedCIImage,
										from: theRect,
										format: thePixelFormat,
										colorSpace: theOutputColorSpace
		) else {
			return nil
		}

		return theCGImage
	}
	
	
// MARK: -
// MARK: For subclass override

	func encodeCommands(
		commandBuffer: MTLCommandBuffer,
		renderPassDescriptor: MTLRenderPassDescriptor,
		frameWidth: UInt,
		frameHeight: UInt,
		quality: GeometryGamesImageQuality)
	{
		//	The GeometryGamesRenderer subclass should override encodeCommands()
		//	to do the actual drawing.
		
		//	If we want the GPU to clear the frame buffer to the background color,
		//	we must submit a command encoder.  An empty one is fine.
		if let theCommandEncoder = commandBuffer.makeRenderCommandEncoder(descriptor: renderPassDescriptor) {
			theCommandEncoder.label = "Geometry Games dummy encoder"
			theCommandEncoder.endEncoding()
		}
	}
}


enum GeometryGamesImageQuality {

	//	During animations, the renderer may want
	//	to strike a balance between image quality
	//	and rendering speed.  For example,
	//	Curved Spaces and Crystal Flight render
	//	progressively more distant objects
	//	using progressively coarser meshes.
	case animation
	
	//	When exporting a still image, the renderer
	//	may want to increase image quality
	//	at the expense of a longer render time.
	//	For example, Curved Spaces and Crystal Flight
	//	draw the whole scene using their best
	//	mesh refinements.
	case export
}


func CreateRenderTarget(
	device: MTLDevice,
	colorBuffer: MTLTexture,
	sampleCount: UInt,					//	typically 4 (for multisampling) or 1 (for single-sampling)
	clearColor: MTLClearColor,
	depthPixelFormat: MTLPixelFormat	//	= MTLPixelFormat.invalid if no depth buffer is needed
) -> MTLRenderPassDescriptor? {

	//	Create a complete render target using the given colorBuffer.
	//	Note that theMultisampleBuffer, theDepthBuffer and theStencilBuffer,
	//	if present, will all be memoryless, so the present function creates
	//	a MTLRenderPassDescriptor without actually creating any new buffers.

	//	Render pass descriptor
	let theRenderPassDescriptor = MTLRenderPassDescriptor()

	//	Color buffer
	guard let theColorAttachmentDescriptor = theRenderPassDescriptor.colorAttachments[0] else {
		return nil
	}
	theColorAttachmentDescriptor.clearColor = clearColor
	theColorAttachmentDescriptor.loadAction = MTLLoadAction.clear	//	always MTLLoadAction.clear for robustness
	if (sampleCount > 1) {
		let theMultisampleTextureDescriptor = MTLTextureDescriptor.texture2DDescriptor(
												pixelFormat: colorBuffer.pixelFormat,
												width: colorBuffer.width,
												height: colorBuffer.height,
												mipmapped: false)
		theMultisampleTextureDescriptor.textureType = MTLTextureType.type2DMultisample
		theMultisampleTextureDescriptor.sampleCount = Int(sampleCount)	//	must match value in pipeline state
		theMultisampleTextureDescriptor.usage = MTLTextureUsage.renderTarget
		theMultisampleTextureDescriptor.storageMode = MTLStorageMode.memoryless
		
		let theMultisampleBuffer = device.makeTexture(descriptor: theMultisampleTextureDescriptor)

		theColorAttachmentDescriptor.texture = theMultisampleBuffer
		theColorAttachmentDescriptor.resolveTexture = colorBuffer
		theColorAttachmentDescriptor.storeAction = MTLStoreAction.multisampleResolve
	}
	else {	//	sampleCount == 1
		theColorAttachmentDescriptor.texture = colorBuffer
		theColorAttachmentDescriptor.storeAction = MTLStoreAction.store
	}
	
	//	Depth buffer
	if (depthPixelFormat != MTLPixelFormat.invalid) {	//	Caller wants depth buffer?
		let theDepthTextureDescriptor = MTLTextureDescriptor.texture2DDescriptor(
											pixelFormat: depthPixelFormat,
											width: colorBuffer.width,
											height: colorBuffer.height,
											mipmapped: false)
		if (sampleCount > 1) {
			theDepthTextureDescriptor.textureType = MTLTextureType.type2DMultisample
			theDepthTextureDescriptor.sampleCount = Int(sampleCount)
		}
		else {	//	sampleCount == 1
			theDepthTextureDescriptor.textureType = MTLTextureType.type2D
			theDepthTextureDescriptor.sampleCount = 1
		}
		theDepthTextureDescriptor.usage = MTLTextureUsage.renderTarget
		theDepthTextureDescriptor.storageMode = MTLStorageMode.memoryless
		let theDepthBuffer = device.makeTexture(descriptor: theDepthTextureDescriptor)
		
		guard let theDepthAttachmentDescriptor = theRenderPassDescriptor.depthAttachment else {
			return nil
		}
		theDepthAttachmentDescriptor.texture = theDepthBuffer
		theDepthAttachmentDescriptor.clearDepth = 1.0
		theDepthAttachmentDescriptor.loadAction = MTLLoadAction.clear
		theDepthAttachmentDescriptor.storeAction = MTLStoreAction.dontCare
	}
	
	//	Stencil buffer (currently unused)
//	if (false) {	//	No current Geometry Games app needs a stencil buffer.
//					//	(If one does someday need a stencil buffer,
//					//	consider the possibility of a combined depth-stencil buffer,
//					//	implemented via an MTLDepthStencilDescriptor.)
//
//		let theStencilTextureDescriptor = MTLTextureDescriptor.texture2DDescriptor(
//											pixelFormat: MTLPixelFormat.stencil8,
//											width: colorBuffer.width,
//											height: colorBuffer.height,
//											mipmapped: false)
//		if (sampleCount > 1) {
//			theStencilTextureDescriptor.textureType = MTLTextureType.type2DMultisample
//			theStencilTextureDescriptor.sampleCount = Int(sampleCount)
//		}
//		else {	//	sampleCount == 1
//			theStencilTextureDescriptor.textureType = MTLTextureType.type2D
//			theStencilTextureDescriptor.sampleCount = 1
//		}
//		theStencilTextureDescriptor.usage = MTLTextureUsage.renderTarget
//		theStencilTextureDescriptor.storageMode = MTLStorageMode.memoryless
//		let theStencilBuffer = device.makeTexture(descriptor: theStencilTextureDescriptor)
//
//		guard let theStencilAttachmentDescriptor = theRenderPassDescriptor.stencilAttachment else {
//		   return nil
//		}
//		theStencilAttachmentDescriptor.texture = theStencilBuffer
//		theStencilAttachmentDescriptor.clearStencil = 0
//		theStencilAttachmentDescriptor.loadAction = MTLLoadAction.clear
//		theStencilAttachmentDescriptor.storeAction = MTLStoreAction.dontCare
//	}
	
	//	All done!
	return theRenderPassDescriptor
}


//	Create a color buffer for offscreen rendering,
//	typically in response to a Copy Image or Save Image command.
//
func CreateOffscreenColorBuffer(
	device: MTLDevice,
	colorPixelFormat: MTLPixelFormat,
	widthPx: UInt,	//	in pixels, not points
	heightPx: UInt	//	in pixels, not points
) -> MTLTexture? {

	let theColorBufferDescriptor = MTLTextureDescriptor.texture2DDescriptor(
									pixelFormat: colorPixelFormat,
									width: Int(widthPx),
									height: Int(heightPx),
									mipmapped: false)
	theColorBufferDescriptor.textureType = MTLTextureType.type2D
		//	Be sure to include MTLTextureUsage.shaderRead
		//	so Core Image can read the pixels afterwards.
	theColorBufferDescriptor.usage = [MTLTextureUsage.renderTarget, MTLTextureUsage.shaderRead]
	theColorBufferDescriptor.storageMode = MTLStorageMode.private
	let theColorBuffer = device.makeTexture(descriptor: theColorBufferDescriptor)
	
	return theColorBuffer
}


func GetMaxFramebufferSizeOnDevice(_ device: MTLDevice) -> UInt {

	let theMaxTextureSize: UInt

	//	A color buffer gets created in Metal as a texture, via the call
	//
	//		device.makeTexture(descriptor: theColorBufferDescriptor)
	//
	//	so the maximum texture size also tells the maximum framebuffer size.

	if
		device.supportsFamily(MTLGPUFamily.apple6)	//	Apple A13
	 || device.supportsFamily(MTLGPUFamily.apple5)	//	Apple A12
	 || device.supportsFamily(MTLGPUFamily.apple4)	//	Apple A11
	 || device.supportsFamily(MTLGPUFamily.apple3)	//	Apple A9, A10
	{
		theMaxTextureSize = 16384
	}
	else
	if  device.supportsFamily(MTLGPUFamily.apple2)	//	Apple A8
	 || device.supportsFamily(MTLGPUFamily.apple1)	//	Apple A7
	{
		theMaxTextureSize =  8192
	}
	else
	if  device.supportsFamily(MTLGPUFamily.mac2)		//	unspecified Mac GPUs
	 || device.supportsFamily(MTLGPUFamily.mac1)		//	unspecified Mac GPUs
	{
		theMaxTextureSize = 16384
	}
	else
	if  device.supportsFamily(MTLGPUFamily.macCatalyst2)
	 || device.supportsFamily(MTLGPUFamily.macCatalyst1)
	{
		//	As of 3 November 2019, there seems to be no documentation
		//	for the Mac Catalyst GPU families.
		//	So let's just try to make a safe-ish guess,
		//	and hope for the best.
		theMaxTextureSize =  8192	//	UNDOCUMENTED GUESS
	}
	else
	if  device.supportsFamily(MTLGPUFamily.common3)
	 || device.supportsFamily(MTLGPUFamily.common2)
	 || device.supportsFamily(MTLGPUFamily.common1)
	{
		//	If we're running on something other than an Apple Silicon GPU
		//	or an Intel Mac GPU, then we'll end up here.
		//	Alas as of 3 November 2019, the documentation
		//	for the three "Common" GPU families gives
		//	feature availability but no implementation limits.
		//	Again, let's take a guess and hope for the best.
		theMaxTextureSize =  8192	//	UNDOCUMENTED GUESS
	}
	else
	{
		//	We should never get to this point,
		//	under any circumstances.
		exit(1)
	}
	
	return theMaxTextureSize
}
